cap log close

/*******************************************************************************

						Real-time 10 year annual average

This program creates real-time 10 year annual average growth rate series used in
Figure7_SPFproductivity.xlsx

Data originally downloaded from ALFRED. 
https://alfred.stlouisfed.org/

Go to ALFRED and download the series Nonfarm Business Sector: Real output Per 
Hour of All Persons. Download all the data. This program cycles through each
release and keeps the first release from each year and then calculates the
10-year average log change for each release(variable).

The program aggregates quarterly data to annual data by averaging.

Then it calculates the 10-year average annual growth rate as

100*(ln(value_year/value_(year-10)))/10
 

Note: there will be
missings for pre-1990 data points since sometimes the first release from each
year does not have Q4 data for the previous year. This is spot checked in the 
"Drop non-applicable releases" section below for 1994, 1996, and 1992. If you
need data for before then 1990, simply find the first release for that year
that does contain data for all 4 quarters of the previous year and keep that one
instead of the first release in that year as done in the "Drop non-applicable 
releases" section.

*******************************************************************************/
					** Input and Calibration **

* Output directory (saves an excel file in following location with the following name)
local output "..\Output"

* Input directory
local input "..\Input"

* Date range for the desired data
local firstYear = 1968
local lastYear = 2017					

********************************************************************************
clear
set more off

* Import excel source sheets, save in Stata format
import excel "`input'\Output Per Hour Historical ALFRED.xlsx", sheet("Quarterly_1") firstrow clear
save "`input'\Quarterly_1.dta", replace
import excel "`input'\Output Per Hour Historical ALFRED.xlsx", sheet("Quarterly_2") firstrow clear
save "`input'\Quarterly_2.dta", replace


* Merge the excel sheets together into one Stata file
use "`input'\Quarterly_1.dta", clear
merge 1:1 observation_date using "`input'\Quarterly_2.dta"
drop _merge

* Find which columns to keep (first from each year)
local flag1 = 0
local flag2 = 0
local keepList
local dropList
local curYear
foreach v of varlist OPHNFB_19680527-OPHNFB_20170202 {
	
	if `flag2'==0 {
		local curYear = substr("`v'",8,4)
		local flag2=1
		continue
	}
	
	if `flag1'==0 {
		local curYear = substr("`v'",8,4)
		local year = substr("`v'",8,4)
		local flag1=1
	}
	else {
		local year = substr("`v'",8,4)
		//di "`year'"
	}
	if `year'==`curYear' {
		//local keepList `keepList' `v'
		local dropList `dropList' `v'
	}
	else {
		local flag1=0
	}
}

di "`dropList'"

* Drop non-applicable releases
foreach v in `dropList' {
	* Spot checks - sometimes the 1st release from each year does not have the data from the 4th quarter of the previous year. 
	* So we use the 2nd release from that year. 
	if "`v'"=="OPHNFB_19940615" {
		drop OPHNFB_19940208 //first 1994 variable does not have an observation 10 years prior. Use OPHNFB_19940615 instead.
	}
	else if "`v'"=="OPHNFB_19960306" {
		drop OPHNFB_19960208
	}
	else if "`v'"=="OPHNFB_19920205" {
		drop OPHNFB_19920103
	}
	
	else {
		drop `v'
	}
}

* Loop through the variables last to first
local vars
foreach v of varlist OPHNFB_19680527 - OPHNFB_20170202 {
	local vars `vars' `v'
}
di "`vars'"
local count = wordcount("`vars'")


* Calculate average growth rate over the last 10 years
local index = _N
local flag = 0

forval i=`count'(-1)1 {
	local v = word("`vars'",`i')
	di "`v'"
	
	local year = substr("`v'",7,5)
	di "`year'"
	if `flag'==0 {
		local flag = 1
	}
	else {
		local index = `index'-4
	}
	
	di `index'
	local bIndex = `index'-40
	di `bIndex'
	
	local endYear = (`v'[`index']+`v'[`index'-1]+`v'[`index'-2]+`v'[`index'-3])/4
	local begYear = (`v'[`bIndex']+`v'[`bIndex'-1]+`v'[`bIndex'-2]+`v'[`bIndex'-3])/4
	local value = 100*(ln(`endYear'/`begYear'))/10
			
	gen `year' = `value'
}



keep _`lastYear'-_`firstYear'
keep if _n==1
gen id = 1
reshape long _, i(id) j(year)
drop id
rename _ realTime_10yr

export excel using "`output'\realTime_10yr.xlsx", firstrow(variables) replace
